<?php
/**
 * Cron Job Handler
 * This file should be run periodically (e.g., every hour) via cron job
 * 
 * To set up cron job:
 * Add this line to crontab (crontab -e):
 * 0 * * * * /usr/bin/php /path/to/TaskManagement/cron_jobs.php
 */

require_once 'config.php';

$conn = getDBConnection();

// Update overdue tasks
$conn->query("UPDATE tasks SET status = 'overdue' WHERE status IN ('pending', 'in_progress') AND due_date < NOW() AND due_date IS NOT NULL");

// Create notifications for tasks due soon (within 1 hour)
$soon_tasks = $conn->query("SELECT t.*, u.id as user_id FROM tasks t 
    INNER JOIN users u ON t.assigned_to = u.id 
    WHERE t.status IN ('pending', 'in_progress') 
    AND t.due_date BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 1 HOUR)
    AND NOT EXISTS (
        SELECT 1 FROM notifications n 
        WHERE n.task_id = t.id 
        AND n.notification_type = 'task_due_soon'
        AND n.is_read = 0
    )")->fetch_all(MYSQLI_ASSOC);

foreach ($soon_tasks as $task) {
    $stmt = $conn->prepare("INSERT INTO notifications (user_id, task_id, notification_type, message) VALUES (?, ?, 'task_due_soon', ?)");
    $message = "Task '{$task['task_name']}' is due soon!";
    $stmt->bind_param("iis", $task['user_id'], $task['id'], $message);
    $stmt->execute();
}

// Handle recurring tasks that need to be created
$recurring_tasks = $conn->query("SELECT * FROM tasks 
    WHERE task_type = 'recurring' 
    AND status = 'completed' 
    AND next_recurrence_date IS NOT NULL 
    AND next_recurrence_date <= NOW()")->fetch_all(MYSQLI_ASSOC);

foreach ($recurring_tasks as $task) {
    // Calculate next due date
    $next_due = date('Y-m-d H:i:s', strtotime("+{$task['recurrence_interval']} days"));
    $next_recurrence = date('Y-m-d H:i:s', strtotime("+{$task['recurrence_interval']} days"));
    
    // Create new recurring task instance
    $stmt = $conn->prepare("INSERT INTO tasks (task_name, task_description, task_duration, task_type, recurrence_interval, priority, assigned_to, created_by, due_date, next_recurrence_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssisisisss", 
        $task['task_name'], 
        $task['task_description'], 
        $task['task_duration'], 
        $task['task_type'], 
        $task['recurrence_interval'], 
        $task['priority'], 
        $task['assigned_to'], 
        $task['created_by'], 
        $next_due, 
        $next_recurrence
    );
    $stmt->execute();
    
    $new_task_id = $conn->insert_id;
    
    // Update next_recurrence_date of original task
    $update_stmt = $conn->prepare("UPDATE tasks SET next_recurrence_date = ? WHERE id = ?");
    $update_stmt->bind_param("si", $next_recurrence, $task['id']);
    $update_stmt->execute();
    
    // Create notification
    if ($task['assigned_to']) {
        $notif_stmt = $conn->prepare("INSERT INTO notifications (user_id, task_id, notification_type, message) VALUES (?, ?, 'task_recurred', ?)");
        $message = "Recurring task: {$task['task_name']}";
        $notif_stmt->bind_param("iis", $task['assigned_to'], $new_task_id, $message);
        $notif_stmt->execute();
    }
}

// Deduct points for overdue incomplete tasks (run daily at midnight)
$current_hour = (int)date('H');
if ($current_hour === 0) {
    $overdue_tasks = $conn->query("SELECT DISTINCT t.id, t.assigned_to, t.task_name 
        FROM tasks t 
        WHERE t.status = 'overdue' 
        AND t.assigned_to IS NOT NULL
        AND DATE(t.due_date) < CURDATE()
        AND NOT EXISTS (
            SELECT 1 FROM points_history ph 
            WHERE ph.task_id = t.id 
            AND ph.reason = 'Task incomplete (overdue)'
            AND DATE(ph.created_at) = CURDATE()
        )")->fetch_all(MYSQLI_ASSOC);
    
    foreach ($overdue_tasks as $task) {
        // Deduct points
        $points = POINTS_INCOMPLETE;
        $points_stmt = $conn->prepare("UPDATE users SET points = points + ? WHERE id = ?");
        $points_stmt->bind_param("di", $points, $task['assigned_to']);
        $points_stmt->execute();
        
        // Record points history
        $history_stmt = $conn->prepare("INSERT INTO points_history (user_id, task_id, points, reason) VALUES (?, ?, ?, 'Task incomplete (overdue)')");
        $history_stmt->bind_param("iid", $task['assigned_to'], $task['id'], $points);
        $history_stmt->execute();
        
        // Create notification
        $notif_stmt = $conn->prepare("INSERT INTO notifications (user_id, task_id, notification_type, message) VALUES (?, ?, 'points_deducted', ?)");
        $message = "Points deducted for overdue task: {$task['task_name']}";
        $notif_stmt->bind_param("iis", $task['assigned_to'], $task['id'], $message);
        $notif_stmt->execute();
    }
}

echo "Cron job completed at " . date('Y-m-d H:i:s') . "\n";
?>
