<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo();
$conn = getDBConnection();

// Get user statistics
$stats = [];
$stats['total_tasks'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ?");
$stats['total_tasks']->bind_param("i", $_SESSION['user_id']);
$stats['total_tasks']->execute();
$stats['total_tasks'] = $stats['total_tasks']->get_result()->fetch_assoc()['count'];

$stats['completed_tasks'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ? AND status = 'completed'");
$stats['completed_tasks']->bind_param("i", $_SESSION['user_id']);
$stats['completed_tasks']->execute();
$stats['completed_tasks'] = $stats['completed_tasks']->get_result()->fetch_assoc()['count'];

$stats['pending_tasks'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ? AND status IN ('pending', 'in_progress')");
$stats['pending_tasks']->bind_param("i", $_SESSION['user_id']);
$stats['pending_tasks']->execute();
$stats['pending_tasks'] = $stats['pending_tasks']->get_result()->fetch_assoc()['count'];

$completion_percentage = $stats['total_tasks'] > 0 ? round(($stats['completed_tasks'] / $stats['total_tasks']) * 100, 2) : 0;

// Today's statistics for user
$stats['today_pending'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ? AND status IN ('pending', 'in_progress') AND DATE(created_at) = CURDATE()");
$stats['today_pending']->bind_param("i", $_SESSION['user_id']);
$stats['today_pending']->execute();
$stats['today_pending'] = $stats['today_pending']->get_result()->fetch_assoc()['count'];

$stats['today_completed'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ? AND status = 'completed' AND DATE(completed_at) = CURDATE()");
$stats['today_completed']->bind_param("i", $_SESSION['user_id']);
$stats['today_completed']->execute();
$stats['today_completed'] = $stats['today_completed']->get_result()->fetch_assoc()['count'];

$stats['uptown_today_task'] = $conn->prepare("SELECT COUNT(*) as count FROM tasks WHERE assigned_to = ? AND DATE(created_at) = CURDATE()");
$stats['uptown_today_task']->bind_param("i", $_SESSION['user_id']);
$stats['uptown_today_task']->execute();
$stats['uptown_today_task'] = $stats['uptown_today_task']->get_result()->fetch_assoc()['count'];
$stats['uptown_today_total'] = $stats['total_tasks'];

// Get user's tasks
$filter = $_GET['filter'] ?? 'all';
$priority_filter = $_GET['priority'] ?? 'all';

$query = "SELECT t.*, u.full_name as created_by_name 
    FROM tasks t 
    LEFT JOIN users u ON t.created_by = u.id 
    WHERE t.assigned_to = ?";

if ($filter === 'completed') {
    $query .= " AND t.status = 'completed'";
} elseif ($filter === 'incomplete') {
    $query .= " AND t.status != 'completed'";
}

if ($priority_filter !== 'all') {
    $query .= " AND t.priority = ?";
}

$query .= " ORDER BY 
    CASE t.priority 
        WHEN 'high' THEN 1 
        WHEN 'medium' THEN 2 
        WHEN 'low' THEN 3 
    END,
    t.due_date ASC";

$stmt = $conn->prepare($query);
if ($priority_filter !== 'all') {
    $stmt->bind_param("is", $_SESSION['user_id'], $priority_filter);
} else {
    $stmt->bind_param("i", $_SESSION['user_id']);
}
$stmt->execute();
$tasks = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get notifications for header
$notification_count = getNotificationCount();
$header_notifications = getNotifications($_SESSION['user_id'], 10);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
    <title>My Dashboard - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/theme.js"></script>
    <script src="assets/js/notifications.js"></script>
    <script src="assets/js/mobile-menu.js"></script>
    <script>
        function updateTimers() {
            document.querySelectorAll('.timer').forEach(timer => {
                const dueDate = new Date(timer.dataset.dueDate);
                const now = new Date();
                const diff = dueDate - now;
                
                if (diff <= 0) {
                    timer.textContent = 'Overdue';
                    timer.className = 'timer danger';
                    return;
                }
                
                const days = Math.floor(diff / (1000 * 60 * 60 * 24));
                const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);
                
                if (days > 0) {
                    timer.textContent = `${days}d ${hours}h ${minutes}m`;
                } else if (hours > 0) {
                    timer.textContent = `${hours}h ${minutes}m ${seconds}s`;
                    if (hours < 2) timer.className = 'timer warning';
                } else {
                    timer.textContent = `${minutes}m ${seconds}s`;
                    timer.className = 'timer danger';
                }
            });
        }
        
        setInterval(updateTimers, 1000);
        updateTimers();
    </script>
</head>
<body class="mobile-app">
    <?php include 'mobile_header.php'; ?>

    <!-- Mobile App Stats Grid -->
    <div class="mobile-stats-grid">
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">Total Tasks</div>
            <div class="mobile-stat-value"><?php echo $stats['total_tasks']; ?></div>
        </div>
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">Completed</div>
            <div class="mobile-stat-value">
                <span class="highlight"><?php echo $stats['completed_tasks']; ?></span>
            </div>
        </div>
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">Today pending</div>
            <div class="mobile-stat-value">
                <span class="highlight"><?php echo $stats['today_pending']; ?></span>
                <span class="total">/<?php echo $stats['total_tasks']; ?></span>
            </div>
        </div>
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">Today completed</div>
            <div class="mobile-stat-value">
                <span class="highlight"><?php echo $stats['today_completed']; ?></span>
                <span class="total">/<?php echo $stats['total_tasks']; ?></span>
            </div>
        </div>
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">My Points</div>
            <div class="mobile-stat-value">
                <span class="highlight">+<?php echo number_format($user['points'], 0); ?></span>
            </div>
        </div>
        <div class="mobile-stat-card">
            <div class="mobile-stat-label">Completion Rate</div>
            <div class="mobile-stat-value">
                <span class="highlight"><?php echo $completion_percentage; ?>%</span>
            </div>
        </div>
    </div>

    <!-- Mobile App Action Buttons -->
    <div class="mobile-action-buttons">
        <a href="my_history.php" class="mobile-action-btn">My History</a>
    </div>

    <div class="container">

        <!-- Progress Bar -->
        <div class="card">
            <h3>Overall Progress</h3>
            <div class="progress-bar">
                <div class="progress-fill" style="width: <?php echo $completion_percentage; ?>%">
                    <?php echo $completion_percentage; ?>%
                </div>
            </div>
        </div>

        <!-- Notifications Section -->
        <div id="notifications" class="card">
            <div class="card-header">
                <h2 class="card-title">All Notifications</h2>
            </div>
            <?php 
            $all_notifications = getNotifications($_SESSION['user_id'], 50);
            if (empty($all_notifications)): ?>
                <div class="empty-state">
                    <div class="empty-state-icon">🔔</div>
                    <h3>No notifications</h3>
                    <p>You don't have any notifications yet.</p>
                </div>
            <?php else: ?>
                <?php foreach ($all_notifications as $notif): 
                    $is_unread = $notif['is_read'] == 0;
                    $time_ago = time() - strtotime($notif['created_at']);
                    $time_text = '';
                    if ($time_ago < 60) {
                        $time_text = 'Just now';
                    } elseif ($time_ago < 3600) {
                        $time_text = floor($time_ago / 60) . ' minutes ago';
                    } elseif ($time_ago < 86400) {
                        $time_text = floor($time_ago / 3600) . ' hours ago';
                    } else {
                        $time_text = date('M d, Y H:i', strtotime($notif['created_at']));
                    }
                ?>
                    <div class="notification-item <?php echo $is_unread ? 'unread' : 'read'; ?>" style="margin-bottom: 12px; border-radius: var(--border-radius);" data-notification-id="<?php echo $notif['id']; ?>">
                        <div class="notification-type"><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $notif['notification_type']))); ?></div>
                        <div class="notification-message"><?php echo htmlspecialchars($notif['message']); ?></div>
                        <div class="notification-time"><?php echo $time_text; ?></div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Filters -->
        <div class="filters">
            <form method="GET" action="" style="display: flex; gap: 1rem; flex-wrap: wrap; width: 100%;">
                <div>
                    <label>Status:</label>
                    <select name="filter">
                        <option value="all" <?php echo $filter === 'all' ? 'selected' : ''; ?>>All</option>
                        <option value="completed" <?php echo $filter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                        <option value="incomplete" <?php echo $filter === 'incomplete' ? 'selected' : ''; ?>>Incomplete</option>
                    </select>
                </div>
                <div>
                    <label>Priority:</label>
                    <select name="priority">
                        <option value="all" <?php echo $priority_filter === 'all' ? 'selected' : ''; ?>>All</option>
                        <option value="high" <?php echo $priority_filter === 'high' ? 'selected' : ''; ?>>High</option>
                        <option value="medium" <?php echo $priority_filter === 'medium' ? 'selected' : ''; ?>>Medium</option>
                        <option value="low" <?php echo $priority_filter === 'low' ? 'selected' : ''; ?>>Low</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-primary">Filter</button>
                <a href="dashboard.php" class="btn btn-secondary">Clear</a>
            </form>
        </div>

        <!-- Tasks -->
        <div class="grid">
            <?php if (empty($tasks)): ?>
                <div class="card empty-state">
                    <div class="empty-state-icon">📋</div>
                    <h3>No tasks found</h3>
                    <p>You don't have any tasks matching your filters.</p>
                </div>
            <?php else: ?>
                <?php foreach ($tasks as $task): 
                    $due_date = $task['due_date'] ? new DateTime($task['due_date']) : null;
                    $is_overdue = $due_date && $due_date < new DateTime() && $task['status'] !== 'completed';
                ?>
                    <div class="card task-card <?php echo $task['priority']; ?>-priority <?php echo $task['status'] === 'completed' ? 'completed' : ''; ?>">
                        <div class="task-header">
                            <div>
                                <h3 class="task-title"><?php echo htmlspecialchars($task['task_name']); ?></h3>
                                <p><?php echo htmlspecialchars($task['task_description']); ?></p>
                            </div>
                        </div>
                        <div class="task-meta">
                            <span class="priority-badge priority-<?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?></span>
                            <span class="status-badge status-<?php echo $task['status']; ?>"><?php echo ucfirst(str_replace('_', ' ', $task['status'])); ?></span>
                            <?php if ($due_date && $task['status'] !== 'completed'): ?>
                                <span class="timer" data-due-date="<?php echo $task['due_date']; ?>">Calculating...</span>
                            <?php endif; ?>
                        </div>
                        <div style="margin-top: 1rem;">
                            <p><strong>Duration:</strong> <?php echo $task['task_duration']; ?> minutes</p>
                            <?php if ($due_date): ?>
                                <p><strong>Due:</strong> <?php echo $due_date->format('M d, Y H:i'); ?></p>
                            <?php endif; ?>
                            <?php if ($task['task_type'] === 'recurring'): ?>
                                <p><strong>Type:</strong> Recurring (Every <?php echo $task['recurrence_interval']; ?> days)</p>
                            <?php endif; ?>
                        </div>
                        <div class="action-buttons" style="margin-top: 1rem;">
                            <a href="view_task.php?id=<?php echo $task['id']; ?>" class="btn btn-primary btn-sm">View Details</a>
                            <?php if ($task['status'] !== 'completed'): ?>
                                <a href="complete_task.php?id=<?php echo $task['id']; ?>" class="btn btn-success btn-sm" onclick="return confirm('Mark this task as completed?')">Complete</a>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
