<?php
require_once 'config.php';
requireAdmin();

$user_id = intval($_GET['id'] ?? 0);
if (!$user_id) {
    header('Location: manage_users.php');
    exit;
}

$conn = getDBConnection();

// Get user details
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();

if (!$user) {
    header('Location: manage_users.php');
    exit;
}

// Get user's tasks
$tasks = $conn->prepare("SELECT * FROM tasks WHERE assigned_to = ? ORDER BY created_at DESC");
$tasks->bind_param("i", $user_id);
$tasks->execute();
$tasks = $tasks->get_result()->fetch_all(MYSQLI_ASSOC);

// Get points history
$points_history = $conn->prepare("SELECT ph.*, t.task_name FROM points_history ph LEFT JOIN tasks t ON ph.task_id = t.id WHERE ph.user_id = ? ORDER BY ph.created_at DESC LIMIT 50");
$points_history->bind_param("i", $user_id);
$points_history->execute();
$points_history = $points_history->get_result()->fetch_all(MYSQLI_ASSOC);

// Get completion history
$completion_history = $conn->prepare("SELECT tch.*, t.task_name FROM task_completion_history tch LEFT JOIN tasks t ON tch.task_id = t.id WHERE tch.user_id = ? ORDER BY tch.completed_at DESC LIMIT 50");
$completion_history->bind_param("i", $user_id);
$completion_history->execute();
$completion_history = $completion_history->get_result()->fetch_all(MYSQLI_ASSOC);

// Calculate statistics
$total_tasks = count($tasks);
$completed_tasks = count(array_filter($tasks, fn($t) => $t['status'] === 'completed'));
$completion_percentage = $total_tasks > 0 ? round(($completed_tasks / $total_tasks) * 100, 2) : 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
    <title>User Details - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/theme.js"></script>
    <script src="assets/js/notifications.js"></script>
    <script src="assets/js/mobile-menu.js"></script>
</head>
<body class="mobile-app">
    <?php include 'mobile_header.php'; ?>

    <div class="container">
        <!-- User Info -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">User Information</h2>
            </div>
            <div class="grid-2">
                <div>
                    <p><strong>Username:</strong> <?php echo htmlspecialchars($user['username']); ?></p>
                    <p><strong>Full Name:</strong> <?php echo htmlspecialchars($user['full_name']); ?></p>
                    <p><strong>Email:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
                    <p><strong>Role:</strong> <?php echo ucfirst($user['role']); ?></p>
                </div>
                <div>
                    <p><strong>Total Points:</strong> <?php echo number_format($user['points'], 2); ?></p>
                    <p><strong>Total Tasks:</strong> <?php echo $total_tasks; ?></p>
                    <p><strong>Completed Tasks:</strong> <?php echo $completed_tasks; ?></p>
                    <p><strong>Completion Rate:</strong> <?php echo $completion_percentage; ?>%</p>
                </div>
            </div>
            <div class="progress-bar" style="margin-top: 1rem;">
                <div class="progress-fill" style="width: <?php echo $completion_percentage; ?>%">
                    <?php echo $completion_percentage; ?>%
                </div>
            </div>
        </div>
        
        <!-- Tasks -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Assigned Tasks</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Task Name</th>
                            <th>Priority</th>
                            <th>Status</th>
                            <th>Due Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($tasks)): ?>
                            <tr>
                                <td colspan="5" class="empty-state">No tasks assigned</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($tasks as $task): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($task['task_name']); ?></td>
                                    <td><span class="priority-badge priority-<?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?></span></td>
                                    <td><span class="status-badge status-<?php echo $task['status']; ?>"><?php echo ucfirst(str_replace('_', ' ', $task['status'])); ?></span></td>
                                    <td><?php echo $task['due_date'] ? date('M d, Y H:i', strtotime($task['due_date'])) : 'N/A'; ?></td>
                                    <td>
                                        <a href="view_task.php?id=<?php echo $task['id']; ?>" class="btn btn-primary btn-sm">View</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Points History -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Points History</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Points</th>
                            <th>Reason</th>
                            <th>Task</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($points_history)): ?>
                            <tr>
                                <td colspan="4" class="empty-state">No points history</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($points_history as $ph): ?>
                                <tr>
                                    <td><?php echo date('M d, Y H:i', strtotime($ph['created_at'])); ?></td>
                                    <td style="color: <?php echo $ph['points'] > 0 ? 'green' : 'red'; ?>; font-weight: bold;">
                                        <?php echo $ph['points'] > 0 ? '+' : ''; ?><?php echo number_format($ph['points'], 2); ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($ph['reason']); ?></td>
                                    <td><?php echo htmlspecialchars($ph['task_name'] ?? 'N/A'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Completion History -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Task Completion History</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Task</th>
                            <th>Points Earned</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($completion_history)): ?>
                            <tr>
                                <td colspan="3" class="empty-state">No completion history</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($completion_history as $ch): ?>
                                <tr>
                                    <td><?php echo date('M d, Y H:i', strtotime($ch['completed_at'])); ?></td>
                                    <td><?php echo htmlspecialchars($ch['task_name'] ?? 'N/A'); ?></td>
                                    <td style="color: green; font-weight: bold;">+<?php echo number_format($ch['points_earned'], 2); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</body>
</html>
