<?php
require_once 'config.php';
requireLogin();

$task_id = intval($_GET['id'] ?? 0);
if (!$task_id) {
    header('Location: ' . (isAdmin() ? 'admin_dashboard.php' : 'dashboard.php'));
    exit;
}

$conn = getDBConnection();
$stmt = $conn->prepare("SELECT t.*, u.full_name as assigned_to_name, u2.full_name as created_by_name 
    FROM tasks t 
    LEFT JOIN users u ON t.assigned_to = u.id 
    LEFT JOIN users u2 ON t.created_by = u2.id 
    WHERE t.id = ?");
$stmt->bind_param("i", $task_id);
$stmt->execute();
$task = $stmt->get_result()->fetch_assoc();

if (!$task) {
    header('Location: ' . (isAdmin() ? 'admin_dashboard.php' : 'dashboard.php'));
    exit;
}

// Check if user has access (admin or assigned user)
if (!isAdmin() && $task['assigned_to'] != $_SESSION['user_id']) {
    header('Location: dashboard.php');
    exit;
}

// Get task notes
$notes_stmt = $conn->prepare("SELECT tn.*, u.full_name as author_name FROM task_notes tn LEFT JOIN users u ON tn.user_id = u.id WHERE tn.task_id = ? ORDER BY tn.created_at DESC");
$notes_stmt->bind_param("i", $task_id);
$notes_stmt->execute();
$notes = $notes_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get task attachments
$attachments_stmt = $conn->prepare("SELECT ta.*, u.full_name as uploaded_by FROM task_attachments ta LEFT JOIN users u ON ta.user_id = u.id WHERE ta.task_id = ? ORDER BY ta.uploaded_at DESC");
$attachments_stmt->bind_param("i", $task_id);
$attachments_stmt->execute();
$attachments = $attachments_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Handle note submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_note'])) {
    $note_text = trim($_POST['note_text'] ?? '');
    if (!empty($note_text)) {
        $note_stmt = $conn->prepare("INSERT INTO task_notes (task_id, user_id, note_text) VALUES (?, ?, ?)");
        $note_stmt->bind_param("iis", $task_id, $_SESSION['user_id'], $note_text);
        $note_stmt->execute();
        logAudit('note_added', 'task', $task_id, 'Note added to task');
        header('Location: view_task.php?id=' . $task_id);
        exit;
    }
}

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['attachment'])) {
    $file = $_FILES['attachment'];
    if ($file['error'] === UPLOAD_ERR_OK && $file['size'] <= MAX_FILE_SIZE) {
        $file_name = basename($file['name']);
        $file_ext = pathinfo($file_name, PATHINFO_EXTENSION);
        $new_file_name = uniqid() . '_' . time() . '.' . $file_ext;
        $file_path = UPLOAD_DIR . $new_file_name;
        
        if (move_uploaded_file($file['tmp_name'], $file_path)) {
            $attach_stmt = $conn->prepare("INSERT INTO task_attachments (task_id, user_id, file_name, file_path, file_size) VALUES (?, ?, ?, ?, ?)");
            $attach_stmt->bind_param("iissi", $task_id, $_SESSION['user_id'], $file_name, $file_path, $file['size']);
            $attach_stmt->execute();
            logAudit('attachment_added', 'task', $task_id, "Attachment added: {$file_name}");
            header('Location: view_task.php?id=' . $task_id);
            exit;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
    <title>View Task - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/theme.js"></script>
    <script src="assets/js/notifications.js"></script>
    <script src="assets/js/mobile-menu.js"></script>
    <script>
        function updateTimer() {
            const timer = document.getElementById('task_timer');
            if (!timer || !timer.dataset.dueDate) return;
            
            const dueDate = new Date(timer.dataset.dueDate);
            const now = new Date();
            const diff = dueDate - now;
            
            if (diff <= 0) {
                timer.textContent = 'Overdue';
                timer.className = 'timer danger';
                return;
            }
            
            const days = Math.floor(diff / (1000 * 60 * 60 * 24));
            const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((diff % (1000 * 60)) / 1000);
            
            if (days > 0) {
                timer.textContent = `${days}d ${hours}h ${minutes}m`;
            } else if (hours > 0) {
                timer.textContent = `${hours}h ${minutes}m ${seconds}s`;
                if (hours < 2) timer.className = 'timer warning';
            } else {
                timer.textContent = `${minutes}m ${seconds}s`;
                timer.className = 'timer danger';
            }
        }
        
        setInterval(updateTimer, 1000);
        updateTimer();
    </script>
</head>
<body class="mobile-app">
    <?php include 'mobile_header.php'; ?>

    <div class="container">
        <div class="card task-card <?php echo $task['priority']; ?>-priority">
            <div class="task-header">
                <div>
                    <h2 class="task-title"><?php echo htmlspecialchars($task['task_name']); ?></h2>
                    <p><?php echo nl2br(htmlspecialchars($task['task_description'])); ?></p>
                </div>
            </div>
            
            <div class="task-meta">
                <span class="priority-badge priority-<?php echo $task['priority']; ?>"><?php echo ucfirst($task['priority']); ?></span>
                <span class="status-badge status-<?php echo $task['status']; ?>"><?php echo ucfirst(str_replace('_', ' ', $task['status'])); ?></span>
                <?php if ($task['due_date'] && $task['status'] !== 'completed'): ?>
                    <span class="timer" id="task_timer" data-due-date="<?php echo $task['due_date']; ?>">Calculating...</span>
                <?php endif; ?>
            </div>
            
            <div style="margin-top: 1.5rem;">
                <p><strong>Duration:</strong> <?php echo $task['task_duration']; ?> minutes</p>
                <p><strong>Type:</strong> <?php echo ucfirst(str_replace('_', ' ', $task['task_type'])); ?></p>
                <?php if ($task['task_type'] === 'recurring'): ?>
                    <p><strong>Recurrence Interval:</strong> Every <?php echo $task['recurrence_interval']; ?> days</p>
                <?php endif; ?>
                <p><strong>Assigned To:</strong> <?php echo htmlspecialchars($task['assigned_to_name'] ?? 'Unassigned'); ?></p>
                <p><strong>Created By:</strong> <?php echo htmlspecialchars($task['created_by_name']); ?></p>
                <p><strong>Created At:</strong> <?php echo date('M d, Y H:i', strtotime($task['created_at'])); ?></p>
                <?php if ($task['due_date']): ?>
                    <p><strong>Due Date:</strong> <?php echo date('M d, Y H:i', strtotime($task['due_date'])); ?></p>
                <?php endif; ?>
                <?php if ($task['completed_at']): ?>
                    <p><strong>Completed At:</strong> <?php echo date('M d, Y H:i', strtotime($task['completed_at'])); ?></p>
                <?php endif; ?>
            </div>
            
            <?php if ($task['status'] !== 'completed' && $task['assigned_to'] == $_SESSION['user_id']): ?>
                <div style="margin-top: 1.5rem;">
                    <a href="complete_task.php?id=<?php echo $task['id']; ?>" class="btn btn-success" onclick="return confirm('Mark this task as completed?')">Mark as Completed</a>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Notes Section -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Notes</h2>
            </div>
            
            <form method="POST" action="" style="margin-bottom: 1.5rem;">
                <div class="form-group">
                    <label for="note_text">Add Note</label>
                    <textarea id="note_text" name="note_text" rows="3" required></textarea>
                </div>
                <button type="submit" name="add_note" class="btn btn-primary">Add Note</button>
            </form>
            
            <div class="notes-section">
                <?php if (empty($notes)): ?>
                    <p class="empty-state">No notes yet</p>
                <?php else: ?>
                    <?php foreach ($notes as $note): ?>
                        <div class="note-item">
                            <div class="note-author"><?php echo htmlspecialchars($note['author_name']); ?></div>
                            <div><?php echo nl2br(htmlspecialchars($note['note_text'])); ?></div>
                            <div class="note-date"><?php echo date('M d, Y H:i', strtotime($note['created_at'])); ?></div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Attachments Section -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Attachments</h2>
            </div>
            
            <form method="POST" action="" enctype="multipart/form-data" style="margin-bottom: 1.5rem;">
                <div class="form-group">
                    <label for="attachment">Upload Attachment (Max 5MB)</label>
                    <input type="file" id="attachment" name="attachment" accept="*/*">
                </div>
                <button type="submit" class="btn btn-primary">Upload</button>
            </form>
            
            <?php if (empty($attachments)): ?>
                <p class="empty-state">No attachments yet</p>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>File Name</th>
                            <th>Size</th>
                            <th>Uploaded By</th>
                            <th>Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($attachments as $attach): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($attach['file_name']); ?></td>
                                <td><?php echo number_format($attach['file_size'] / 1024, 2); ?> KB</td>
                                <td><?php echo htmlspecialchars($attach['uploaded_by']); ?></td>
                                <td><?php echo date('M d, Y H:i', strtotime($attach['uploaded_at'])); ?></td>
                                <td>
                                    <a href="<?php echo htmlspecialchars($attach['file_path']); ?>" class="btn btn-primary btn-sm" download>Download</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
